#!/usr/bin/env python3
# wrtten by T. Nakamura, KEK, 2024-03-27, 2019-03-17
#################
### example of input python file
#################
#--BEGIN input file --
# turnL = [ 1,2,3,4,5,6,7,8,9 ]    # previous turn number of input 
#                                  (this case, the position data of 1st to 9-th previous turns are input )
# tune0 = {"nu":0,    "gain":0,             "order":0 } # dictionaly for the constraint of tune1 
# tune1 = {"nu":0.15, "gain":1, "zeta":-90, "order":1 } # dictionaly for the constraint of tune2 
# tuneL = [ tune0, tune1 ]                                             # list of tune data dictionary 
#--END  input file --
#################
# parameters of each dictionary are
#  nu     : target tune
#  gain   : gain at target tune 
#  zeta   : phase at target tune in degree
#  order  : order of Tylor expansion for tune shift around target tune
#      order = 0 : filter is just band pass filter, 
#      order = 1 : with "flat gain and phase region around target tune 
#      order = 2 : with "more flat gain and phase region around target tune 
#         ...
#      G(nu) = gain(nu) * exp( i*zeta(nu) ) 
#      d^n G(nu)/dnu^n = 0 for n = 1,2, .., order 
#          (flat gain and phase around nu  with order >0 )
#  for nu=0
#     no phase parameters are necessary (zeta)
##################################

import sys, math, cmath, random, re, os, os.path; from math import *; from cmath import *  
import numpy as np 
import importlib
#  load input module
argvs = sys.argv
argc  = len( argvs ) - 1
if argc != 1  :
   print("    usage : ", argvs[0],  " input_python_file" )
   quit()
infile = argvs[1]
loader = importlib.machinery.SourceFileLoader('in_put', infile )
in_put = loader.load_module()

def main( tuneL, turnL ) : #  be careful, tuNEL and  tuRNL
# --- input parameter : start
#    print("# STEP 1")
# --- input parameter : end
    print( "# turnL = ", turnL )
    for tune in tuneL : 
        print( "# tune = ", tune )
    ntap = len( turnL )
#
# --- get parameters ---
#
    nuL, gainL, zetaL, orderL = dict2data( tuneL )
#
# --- check constraints and taas
#
    nconstr = 0
    for i in range(len(nuL)):
        if nuL[i] == 0 :
           nconstr = nconstr + orderL[i] 
        if nuL[i] != 0 :
           nconstr = nconstr + 2*orderL[i] 
    if len(turnL) < nconstr :
        print("ERROR: number of taps < anumber of constraints") 
        print("   ntap   nconstr  = ", ntap, nconstr )
        quit()
#
# --- make phi[m,k] ---
#
    phi = make_phi( turnL, nuL )
#
# --- calculate coefficients
#
    print("## ----- calculate coef  -------")
    coefL = calc_coef( phi, orderL, gainL, zetaL )
    sum2 = 0
    for i in range( len(coefL) ) :
        sum2 = sum2 + coefL[i]**2
#
# --- print coefficients 
#
    print( len( coefL ) )
    print("# %20s   %15s" % ( "coefficients", "turn") )
    for i in range( len(turnL) ):
        print("%20.10f   %15.5f" % ( coefL[i], turnL[i] ) )
    print("## power_pass = sum(coef^2) = ", sum2 )

def calc_coef( phi, orderL, gainL, zetaL ) :
    def calc_fmu_gmu( phi, orderL, gainL, zetaL ) :
        def f0l( l, phi ) :  
            return phi**l    # In python, 0**0 = 1
        def flc( l, phi ) :
            return phi**l * np.cos(phi)
        def fls( l, phi ) :
            return phi**l * np.sin(phi)
        def calc_nmu( phi, orderL ) :
            nmu = 0
            nnu = len( phi[:,0] )
            for m in range( nnu ) :
               if np.count_nonzero( phi[m,:] ) == 0 :  # tune == 0 
                  nmu = nmu + 1*( orderL[m] + 1 )
               else :
                  nmu = nmu + 2*( orderL[m] + 1 )
            return nmu
        ntap = len( phi[0,:] ) 
        nnu  = len( phi[:,0] )
        nmu = calc_nmu( phi, orderL )
        fmu = np.zeros( (nmu, ntap) )
        gmu = np.zeros( nmu )
        mu = 0
        for m in range( nnu ) :
            if np.count_nonzero( phi[m,:] ) == 0 :  # tune == 0 
               for l in range( orderL[m] + 1 ) :
                   for k in range( ntap ) :
                       fmu[mu,k] = f0l( l, k )
                   if l == 0 :
                      gmu[mu] = gainL[m]
                   mu = mu+1
            else :                                 # tune != 0 
               for l in range( orderL[m] + 1 ) :
                   for k in range( ntap ) :
                       fmu[mu,k] = flc( l, phi[m,k] )
                   if l == 0 :
                      gmu[mu] = gainL[m] * np.cos( zetaL[m] ) 
                   mu = mu+1
                   for k in range( ntap ) :
                       fmu[mu,k] = fls( l, phi[m,k] )
                   if l == 0 :
                      gmu[mu] = gainL[m] * np.sin( zetaL[m] ) 
                   mu = mu+1
        return fmu, gmu, mu, nmu
    def calc_FF( fmu, mu ) :
        FF = np.zeros( ( mu,mu ),dtype=float)
        ntap = len( fmu[0,:] )
        for mu1 in range(mu) :
            for mu2 in range(mu) :
                FF[mu1,mu2] = 0
                for k in range(ntap) :
                    FF[mu1,mu2] = FF[mu1,mu2] + fmu[mu1,k]*fmu[mu2,k] 
        return FF
    fmu, gmu, mu, nmu = calc_fmu_gmu( phi, orderL, gainL, zetaL )
    FF = calc_FF( fmu, mu )
    detFF = np.linalg.det( FF ) 
    if detFF == 0 :
       print("ERROR : det(FF) = 0")
       quit()
    FFinv = np.linalg.inv( FF ) 
    FFinvF = np.dot( FFinv, fmu )
    coef = np.dot( gmu, FFinvF )
    return coef 

def make_phi( turnL, nuL ):
    ntap = len( turnL )
    nnu  = len( nuL )
    phi = np.zeros( (nnu, ntap) )
    for m in range( nnu ) :
        for k in range( ntap ) :
            phi[m,k]  = -2*pi*nuL[m]*turnL[k] 
    return phi
     

def dict2data( tuneL ) :
    nuL, gainL, zetaL, orderL = [], [], [], []
    for tune in tuneL :
        nu_m    = tune['nu']
        gain_m  = tune['gain']
        order_m = tune['order']
        if nu_m == 0 :
           zeta_m  = 0
        else :
           zeta_m   = float( tune['zeta'] )/180.*pi
        nuL.append( nu_m )
        gainL.append( gain_m )
        zetaL.append( zeta_m )
        orderL.append( order_m )
    return nuL, gainL, zetaL, orderL

if __name__ == '__main__' :
    main( in_put.tuneL, in_put.turnL )

